% "Survey density forecast comparison in small samples" 
% by Laura Coroneo, Fabrizio Iacone and Fabio Profumo, 
% International Journal of Forecasting, 2024

% Monte Carlo
% This code replicates Figure 1, the finite sample power of the EPA test.

clc;
clear;

TT = [40 80];               % sample sizes
M = 10000;                  % number of simulations
tsize=0.05;                 % test size
q=0;                        % no dependence
C = 0:0.25:4;               % Max c in X-axis of plots = 4
alpha = 1;
aa = [1/3 1/2 1];           % wce bands exponent (T^aa)
bb = [1/4 1/3 1/2 2/3];     % wpe bands exponent (T^bb)

%% initialise

DM_WCE = NaN(length(C), M, length(aa), length(TT));
reject_fix_b= NaN(length(C), M, length(aa), length(TT));
reject_wce_N = NaN(length(C), M, length(aa), length(TT));
reject_wce_sad = NaN([length(C),M, length(aa), length(TT)]);

DM_WPE = NaN(length(C), M, length(bb), length(TT));
reject_fix_m = NaN(length(C), M, length(bb), length(TT));
reject_wpe_N = NaN(length(C), M, length(bb), length(TT));
reject_wpe_sad = NaN([length(C),M, length(bb), length(TT)]);


for  ts= 1:length(TT)
    clear beta A B a b e1 e2 d ww

    T = TT(ts);

    % WCE bands
    wce_bands = floor(T.^aa);

    % WPE bands
    wpe_bands = floor(T.^bb);

    for cc = 1:length(C)

        c = C(cc);

        rng(0,'twister');

        beta = (alpha^2-(3/2*c/sqrt(T)))^(1/2);

        for m=1:M

            %% simulate

            a = unifrnd(0, alpha/(q+1), [T+q 1]);
            b = unifrnd(0, beta/(q+1), [T+q 1]);

            for t = 1+q:T+q

                A(t) = sum(a(t-q:t));
                B(t) = sum(b(t-q:t));


                e1 = [A(t);-A(t);0];
                e2 = [0;-B(t);B(t)];

                d(t,:) =  e1'*e1-e2'*e2;

            end

            d = d(q+1:end);

            %% test


            for w = 1:length(wce_bands)
                ww = wce_bands(w);
                [DM_WCE(cc,m,w,ts), reject_fix_b(cc,m,w,ts)] = dm(d, ww);
            end

            for w =  1:length(wpe_bands)
                ww = wpe_bands(w);
                [~,~,~, DM_WPE(cc,m,w,ts), reject_fix_m(cc,m,w,ts)] = dm(d, w, ww);
            end


        end

    end

    %% Size Aadjusted CV
    quant_wce = squeeze(quantile(DM_WCE(1,:,:,ts), 0.95))';
    quant_wpe = squeeze(quantile(DM_WPE(1,:,:,ts), 0.95))';

    for cc= 1:length(C)
        for m = 1:M
            reject_wce_sad(cc,m,:,ts) = squeeze(DM_WCE(cc,m,:,ts)) > quant_wce';
            reject_wpe_sad(cc,m,:,ts) = squeeze(DM_WPE(cc,m,:,ts)) > quant_wpe';
        end
    end

    %% Saving results in tables

    RESFS(ts).values = table(transpose(C),...
        sum(reject_fix_b(:,:,1,ts),2)./M,...
        sum(reject_fix_b(:,:,2,ts),2)./M,...
        sum(reject_fix_b(:,:,3,ts),2)./M,...
        sum(reject_fix_m(:,:,1,ts),2)./M,...
        sum(reject_fix_m(:,:,2,ts),2)./M,...
        sum(reject_fix_m(:,:,3,ts),2)./M,....
        sum(reject_fix_m(:,:,4,ts),2)./M,...
        'VariableNames',{'c' 'WCET3' 'WCET2' 'WCET' 'WPET4' 'WPET3' 'WPET2' 'WPET23'});


    RESSAD(ts).values = table(transpose(C),...
        sum(reject_wce_sad(:,:,1,ts),2)./M,...
        sum(reject_wce_sad(:,:,2,ts),2)./M,...
        sum(reject_wce_sad(:,:,3,ts),2)./M,...
        sum(reject_wpe_sad(:,:,1,ts),2)./M,...
        sum(reject_wpe_sad(:,:,2,ts),2)./M,...
        sum(reject_wpe_sad(:,:,3,ts),2)./M,....
        sum(reject_wpe_sad(:,:,4,ts),2)./M,...
        'VariableNames',{'c' 'WCET3' 'WCET2' 'WCET' 'WPET4' 'WPET3' 'WPET2' 'WPET23'});
end

%% Make plots
tiledlayout(2,2)

for ts=1:length(TT)
    nexttile

    plot(RESSAD(ts).values{:, 1}, RESSAD(ts).values{:, 3}, '--b', 'LineWidth',2); %1/2 SA
    hold
    plot(RESSAD(ts).values{:, 1}, RESFS(ts).values{:, 3}, 'b', 'LineWidth',2); %1/2
    plot(RESSAD(ts).values{:, 1}, RESSAD(ts).values{:, 4}, '--k', 'LineWidth',2); %T SA
    plot(RESSAD(ts).values{:, 1}, RESFS(ts).values{:, 4}, 'k', 'LineWidth',2); %T
    xlim([0 4])
    xticks(0:c)
    legend ('T^{1/2} SA', 'T^{1/2}', 'T SA', 'T', 'Location','southeast');
    title(['WCE T=' num2str(TT(ts))])
    xlabel('c')

    nexttile

    plot(RESFS(ts).values{:, 1}, RESSAD(ts).values{:, 6}, '--b', 'LineWidth',2); %1/3 SA
    hold
    plot(RESFS(ts).values{:, 1}, RESFS(ts).values{:, 6}, 'b', 'LineWidth',2); %1/3
    plot(RESFS(ts).values{:, 1}, RESSAD(ts).values{:, 5}, '--k', 'LineWidth',2); %1/4 SA
    plot(RESFS(ts).values{:, 1}, RESFS(ts).values{:, 5}, 'k', 'LineWidth',2); %1/4
    xlim([0 4])
    xticks(0:c)
    legend ('T^{1/3} SA','T^{1/3}', 'T^{1/4} SA', 'T^{1/4}', 'Location','southeast');
    title(['WPE T=' num2str(TT(ts))])
    xlabel('c')

end

  